/*
 * Copyright (c) 2019. by 8x8. Inc.
 *   _____      _____
 *  |  _  |    |  _  |
 *   \ V /__  __\ V /   ___ ___  _ __ ___
 *   / _ \\ \/ // _ \  / __/ _ \| '_ ` _ \
 *  | |_| |>  <| |_| || (_| (_) | | | | | |
 *  \_____/_/\_\_____(_)___\___/|_| |_| |_|
 *
 *  All rights reserved.
 *
 *  This software is the confidential and proprietary information
 *  of 8x8 Inc. ("Confidential Information").  You
 *  shall not disclose such Confidential Information and shall use
 *  it only in accordance with the terms of the license agreement
 *  you entered into with 8x8 Inc.
 *
 */
package com._8x8.vcc.sapi;

import com.ning.http.client.AsyncHttpClient;
import com.ning.http.client.AsyncHttpClientConfig;
import com.ning.http.client.Realm;
import com.ning.http.client.resumable.ResumableIOExceptionFilter;
import org.atmosphere.wasync.*;
import org.atmosphere.wasync.Request.TRANSPORT;
import org.atmosphere.wasync.impl.AtmosphereClient;
import org.atmosphere.wasync.impl.AtmosphereSocket;
import org.atmosphere.wasync.impl.DefaultOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.nio.CharBuffer;
import java.util.Date;
import java.util.concurrent.TimeoutException;

/**
 * This is the manin entry point for the application and contains
 * all the socket handling logic
 * @author 8x8 Engineering
 */
@SuppressWarnings({"java:S1604","java:S1905"})
public final class AsyncClient {

    private static final Logger logger = LoggerFactory.getLogger(AsyncClient.class);

    // Number of retry
    private static int retry = 3;

    // System variables
    private static long startTime;
    private static AsyncHttpClient asyncHttpClient;
    private static CommandLineArgs cmd;

    /**
     * Prevent instantiation
     */
    private AsyncClient() {
    }

    /**
     * Subscribe to the SAPI endpoint using the command line arguments
     */
    private static void subscribe() {
        startTime = System.currentTimeMillis();
        logger.info("Subscribing with sapiURL:{}, tenantId:{}, securityToken:{}, subscriptionId:{}, desiredOutputType:{}, subsRequestTimeout:{}msec, started at:{}",
                cmd.getSapiUrl(), cmd.getTenantID(), cmd.getSecurityToken(), cmd.getSubscriptionID(), cmd.getOutputType(), cmd.getSubscriptionTimeout(), startTime);

        // Create new thread
        Thread thread = new Thread(AsyncClient::run);

        // Start the thread
        thread.start();
    }

    /**
     * Reads the response stream from the reader
     *
     * @param reader Stream reader
     * @return Response
     */
    private static String readResponse(Reader reader) {

        // Create new Char Buffer
        final CharBuffer cb = CharBuffer.allocate(cmd.getBufferSize());
        if (reader != null) {
            try {
                // Read characters into a char buffer
                if (reader.read(cb) > -1) {

                    // Flip the char buffer
                    cb.flip();
                }
            }
            catch (final IOException e) {
                logger.error("Reader Error - {}", e.getMessage());
            }
            finally {
                try {
                    reader.close();
                }
                catch (IOException e) {
                    logger.error("Problem closing reader - {}", e.getMessage());
                }
            }
        }
        return cb.toString();
    }

    /**
     * If the retry count has been exhausted, exits the application
     */
    private static void unsubscribe() {
        retry++;
        if (retry >= cmd.getConnectionRetries()) {
            exit();
        }
    }

    /**
     * Exits the application
     */
    private static void exit() {
        System.exit(-1);
    }

    /**
     * Attach a hook to the shutdown to release resources
     * when the user closes the app with CTR+C or similar signal
     */
    public static void attachShutDownHook() {

        // Add a shutdown hook to gracefully close the connection
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            logger.info("Shutting down the client now....");
            try {

                // Close the client
                if (asyncHttpClient != null) {
                    asyncHttpClient.closeAsynchronously();
                }
                logger.info("Shutdown complete");
            }
            catch (final Exception e) {
                logger.error("Socket error - {}", e.getMessage());
            }
        }));
    }

    /**
     * Starts the background async socket listener
     */
    private static void startSocketListeners() {
        try {
            // Send subscribe request
            subscribe();

            // Add shutdown hook
            attachShutDownHook();
        }
        catch (final Exception e) {
            logger.error("Failed to subscribe - {}", e.getMessage());
        }
    }

    /**
     * Background task to listen for events from SAPI
     *
     * NOTE: - Although InteliJ and Sonar will be begging you to replace the
     *         anonymous classes in here with lambdas, resist. The wasync library
     *         makes a lot of use of reflection and that doesn't seem to play well
     *         with lambdas as event handlers
     */
    private static void run() {
        AtmosphereSocket socket;

        // Bind different events on socket
        try {
            // Create new user security realm
            final Realm realm = new Realm.RealmBuilder()
                    .setPrincipal(cmd.getTenantID())
                    .setPassword(cmd.getSecurityToken())
                    .setUsePreemptiveAuth(true)
                    .setScheme(Realm.AuthScheme.BASIC)
                    .build();

            // Create new client builder to pass credentials
            final AsyncHttpClientConfig.Builder ccBuilder = new AsyncHttpClientConfig.Builder()
                    .setRealm(realm)
                    .setAllowPoolingConnections(true)
                    .setRequestTimeout(cmd.getRequestTimeout())
                    .setReadTimeout(210000)
                    .addIOExceptionFilter(new ResumableIOExceptionFilter())
                    .setMaxRequestRetry(cmd.getConnectionRetries());

            // Create async client
            asyncHttpClient = new AsyncHttpClient(ccBuilder.build());

            // Create new client
            final AtmosphereClient atmosphereClient = ClientFactory.getDefault().newClient(AtmosphereClient.class);

            // Create options
            final DefaultOptions options = atmosphereClient.newOptionsBuilder()
                    .runtime(asyncHttpClient)
                    .requestTimeoutInSeconds(cmd.getSubscriptionTimeout())
                    .reconnect(cmd.isReconnect())
                    .runtimeShared(false)
                    .pauseBeforeReconnectInSeconds(60)
                    .build();

            // Create new stream request
            final RequestBuilder<?> requestBuilder = atmosphereClient.newRequestBuilder()
                    .method(Request.METHOD.GET)
                    .transport(TRANSPORT.STREAMING)
                    .trackMessageLength(true)
                    .uri(String.format("%s/TenantUpdates-%s-%s", cmd.getSapiUrl(), cmd.getTenantID(), cmd.getSubscriptionID()))
                    .queryString("tenantId", cmd.getTenantID())
                    .queryString("subsId", cmd.getSubscriptionID())
                    .queryString("desiredOutputType", cmd.getOutputType())
                    .decoder(new Decoder<String, Reader>() {
                        @Override
                        public Reader decode(final Event type, final String s) {
                            return new StringReader(s);
                        }
                    })
                    .encoder(new Encoder<String, Reader>() {
                        @Override
                        public Reader encode(final String s) {
                            return new StringReader(s);
                        }
                    });

            // Create new client socket
            socket = (AtmosphereSocket) atmosphereClient.create(options);

            // Socket opened
            socket.on(Event.OPEN.name(), new Function<Reader>() {
                @Override
                public void on(final Reader reader) {
                    logger.info("Opened [{}]", readResponse(reader));
                }
            });

            // Status message available
            socket.on(Event.OPEN.name(), new Function<String>() {
                @Override
                public void on(final String string) {
                    if (string != null) {
                        logger.info("Status [{}]", string);
                    }
                }
            });

            // Headers available
            socket.on(Event.HEADERS.name(), new Function<String>() {
                @Override
                public void on(final String string) {
                    if (string != null) {
                        logger.info("Connected to SAPI server; Headers [{}]", string);
                    }
                }
            });

            // Message available
            socket.on(Event.MESSAGE.name(), new Function<Reader>() {
                @Override
                public void on(final Reader reader) {
                    logger.info("Message\n{}", readResponse(reader));
                }
            });

            // Socket error
            socket.on(Event.ERROR.name(), new Function<IOException>() {
                @Override
                public void on(final IOException e) {
                    logger.error("IOException - {}", e.getMessage());
                }
            });

            // Connection error
            socket.on(Event.ERROR.name(), new Function<TimeoutException>() {
                @Override
                public void on(final TimeoutException t) {
                    logger.error("Connection timeout - {}", t.getMessage());
                    unsubscribe();
                }
            });

            // Socket closed
            socket.on(Event.CLOSE.name(), new Function<Reader>() {
                @Override
                public void on(final Reader reader) {
                    logger.info("Close [{}", readResponse(reader));
                    long endTime = System.currentTimeMillis();

                    final long timeConsumedInMsec = endTime - startTime;
                    final long timeConsumedInSec = timeConsumedInMsec / 1000;
                    final long timeConsumedInMin = timeConsumedInSec / 60;
                    final long timeConsumedInHour = timeConsumedInMin / 60;

                    logger.info("Listening stopped at:{}, Time consumed:{} msec(s) / {} sec(s) / {} min(s) / {} hour(s)",
                            new Date(endTime - startTime), timeConsumedInMsec, timeConsumedInSec, timeConsumedInMin, timeConsumedInHour);
                    logger.info("Socket connection status: {}", socket.status());

                    // Unsubscribe
                    unsubscribe();
                }
            });

            // Connect to the SAPI endpoint
            socket.open(requestBuilder.build());
        }
        catch (final Exception e) {
            logger.debug("Socket Error - {}", e.getMessage());
        }
    }

    /**
     * Main entry point for the application
     *
     * @param args Arguments
     */
    public static void main(final String[] args) {
        try {
            // Get the command line args
            cmd = new CommandLineArgs();
            cmd.parse(args);

            // Start the background socket listener
            startSocketListeners();
        }
        catch (SapiClientException e) {
            logger.error("Command line error: {}", e.getMessage());
            cmd.printUsage();
        }
    }

}
