/*
 * Copyright (c) 2019. by 8x8. Inc.
 *   _____      _____
 *  |  _  |    |  _  |
 *   \ V /__  __\ V /   ___ ___  _ __ ___
 *   / _ \\ \/ // _ \  / __/ _ \| '_ ` _ \
 *  | |_| |>  <| |_| || (_| (_) | | | | | |
 *  \_____/_/\_\_____(_)___\___/|_| |_| |_|
 *
 *  All rights reserved.
 *
 *  This software is the confidential and proprietary information
 *  of 8x8 Inc. ("Confidential Information").  You
 *  shall not disclose such Confidential Information and shall use
 *  it only in accordance with the terms of the license agreement
 *  you entered into with 8x8 Inc.
 *
 */
package com._8x8.vcc.sapi;

import org.apache.commons.cli.*;

/**
 * Gathers and validates the command line arguments
 */
public class CommandLineArgs {
    private final Options options = new Options();

    private String tenantID;
    private String securityToken;
    private String sapiUrl;
    private String subscriptionID = "8x8engineering";
    private String outputType = "xml";
    private boolean reconnect = true;
    private int requestTimeoutMilliSeconds = 20 * 1000;
    private int subscriptionTimeoutSeconds = 10 * 60;
    private int connectionRetries = 3;
    private int bufferSize = 1024 * 20;

    /**
     * Constructs a holder for the command line arguments
     */
    public CommandLineArgs() {

        // Declare all the possible command line arguments with defaults
        options.addOption("t", "tenant", true, "The tenant ID e.g. halfordsgroupplc01");
        options.addOption("tok", "token", true, "The security token from CM for this tenant");
        options.addOption("s", "subid", true, String.format("Subscription ID to use to identify ourselves to SAPI [default: %s]", subscriptionID));
        options.addOption("url", true, "The SAPI endpoint to connect to e.g. https://vcc-eu9.8x8.com/api/streaming/v1/clientconnect/subscribe");
        options.addOption("n", "no_reconnect", false, String.format("Indicates that the connection should not reconnect if it drops [default: %b]", reconnect));
        options.addOption("sub_timeout", true, String.format("The subscription timeout (the time to stay connected whilst waiting for SAPI to send events) in sec [default: %d]", subscriptionTimeoutSeconds));
        options.addOption("req_timeout", true, String.format("The request timeout on making a connection in ms [default: %d]", subscriptionTimeoutSeconds));
        options.addOption("retries", true, String.format("The number of connection retries to attempt before failing [default: %d]", connectionRetries));
        options.addOption("b", "buffer", true, String.format("The size of the buffer in bytes to receive events [default: %d]", bufferSize));
        options.addOption("o", "output", true, String.format("The output type (XML,JSON) [default: %s]", outputType));
        options.addOption("h", "help", false, "Show this help");
    }

    /**
     * Parses the arguments list and validates their suitability
     *
     * @param args Array of arguments
     * @throws SapiClientException If an argument is badly formed or missing
     */
    public void parse(String... args) throws SapiClientException {

        // Parse the command line into the constituents
        CommandLineParser parser = new DefaultParser();
        try {
            CommandLine cmd = parser.parse(options, args);

            // Ignore everything if help has been asked for
            if (cmd.hasOption("h")) {
                printUsage();
                return;
            }

            // Get the values
            tenantID = cmd.getOptionValue("t");
            securityToken = cmd.getOptionValue("tok");
            subscriptionID = cmd.getOptionValue("s", subscriptionID);
            sapiUrl = cmd.getOptionValue("url");
            reconnect = !cmd.hasOption("n");
            subscriptionTimeoutSeconds = Integer.parseInt(cmd.getOptionValue("sub_timeout", subscriptionTimeoutSeconds + ""));
            requestTimeoutMilliSeconds = Integer.parseInt(cmd.getOptionValue("req_timeout", requestTimeoutMilliSeconds + ""));
            connectionRetries = Integer.parseInt(cmd.getOptionValue("retries", connectionRetries + ""));
            bufferSize = Integer.parseInt(cmd.getOptionValue("n", bufferSize + ""));

            // Tidy up
            if (sapiUrl != null) {
                sapiUrl = sapiUrl.replaceAll("[/\\\\]+$", "");
            }
            if (outputType == null || !outputType.matches("(?is)xml|json")) {
                outputType = "xml";
            }
            outputType = outputType.toLowerCase();
        }
        catch (ParseException e) {
            throw new SapiClientException(e.getMessage());
        }

        // Validate the arguments
        validate();
    }

    /**
     * Validates that the combination of arguments is valid
     *
     * @throws SapiClientException If there is a problem
     */
    private void validate() throws SapiClientException {

        if (tenantID == null || tenantID.isEmpty()) {
            throw new SapiClientException("a tenant must be specified");
        }
        if (securityToken == null || securityToken.isEmpty()) {
            throw new SapiClientException("a security token must be specified");
        }
        if (sapiUrl == null || sapiUrl.isEmpty()) {
            throw new SapiClientException("a SAPI URL must be specified");
        }
    }

    /**
     * Prints out a nice display of all the available options
     */
    public void printUsage() {
        HelpFormatter formatter = new HelpFormatter();
        formatter.setWidth(100);
        formatter.printHelp("sapiclient", options);
    }

    /**
     * Returns the tenant ID e.g. dgfshfhg01
     *
     * @return Tenant ID
     */
    public String getTenantID() {
        return tenantID;
    }

    /**
     * Returns the security token - this is the generated ID found i the Security
     * section of the VCC Configuration Manager
     *
     * @return Token ID
     */
    public String getSecurityToken() {
        return securityToken;
    }

    /**
     * This is the ID tha the client uses to identify itself to SAPI
     * This is a completely arbitrary string but it is used by SAPI in a map to collect events
     * on your behalf during times of disconnection. There is a limit to the number
     * of these queues that SAPI will maintain on behalf of tenants (3) so it is important
     * to use the same one ideally and not mix it with a name the customer may be using
     *
     * @return Subsciption ID e.g. 8x8engineering
     */
    public String getSubscriptionID() {
        return subscriptionID;
    }

    /**
     * This is the full URL to the SAPI endpoint without training slashes
     *
     * @return URL e.g. https://vcc-eu9.8x8.com/api/streaming/v1/clientconnect/subscribe
     */
    public String getSapiUrl() {
        return sapiUrl;
    }

    /**
     * This is the flag tha tells the client to reconnect if the connection drops
     *
     * @return Reconnect
     */
    public boolean isReconnect() {
        return reconnect;
    }

    /**
     * This is the time in milliseconds that the client will wait for a connection to be established
     *
     * @return Timeout on conection
     */
    public int getRequestTimeout() {
        return requestTimeoutMilliSeconds;
    }

    /**
     * This is the length in seconds that the connection will stay open waiting for events
     *
     * @return Time to wait for events
     */
    public int getSubscriptionTimeout() {
        return subscriptionTimeoutSeconds;
    }

    /**
     * Maximum number of retries to attempt before failing
     *
     * @return Retires
     */
    public int getConnectionRetries() {
        return connectionRetries;
    }

    /**
     * The size of the buffer to use for catching events - may need adjusting if you set
     * the subscription timeout to be long, and then remain disconnected for a long time
     * so SAPI builds up a lot of events on your behalf
     *
     * @return Buffer size in bytes
     */
    public int getBufferSize() {
        return bufferSize;
    }

    /**
     * This is th style of output needed - only supports XML or JSON
     *
     * @return Output style
     */
    public String getOutputType() {
        return outputType;
    }
}
